/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0, 0.0,0.0)

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////
mat2 rotz(float angle)
{
    mat2 m;
    m[0][0] = cos(angle); m[0][1] = -sin(angle);
    m[1][0] = sin(angle); m[1][1] = cos(angle);
    return m;
}

float rand(vec2 co){
    return fract(sin(dot(co.xy ,vec2(12.9898,78.233))) * 43758.5453);
}


float fbm(vec2 uv)
{
    float n = (texture2D(iChannel1, uv).r - 0.5) * 0.5;
    n += (texture2D(iChannel1, uv * 2.0).r - 0.5) * 0.5 * 0.5;
    n += (texture2D(iChannel1, uv * 3.0).r - 0.5) * 0.5 * 0.5 * 0.5;
    
   return n + 0.5;
}


// -----------------------------------------------

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    
    vec2 uv = fragCoord.xy / iResolution.xy;
    vec2 _uv = uv;
    uv -= vec2(0.5);
    uv.y /= iResolution.x / iResolution.y;
    vec2 centerUV = uv;
    
    // height variation from fbm
    float variationH = fbm(vec2(iGlobalTime * .3)) * 1.1;
    
    // flame "speed"
    vec2 offset = vec2(0.0, -iGlobalTime * 0.15);
    
    // flame turbulence
    float f = fbm(uv * 0.1 + offset); // rotation from fbm
    float l = max(0.1, length(uv)); // rotation amount normalized over distance
    uv += rotz( ((f - 0.5) / l) * smoothstep(-0.2, .4, _uv.y) * 0.45) * uv;    
    
    // flame thickness
    float flame = 1.3 - length(uv.x) * 5.0;
    
    // bottom of flame 
    float blueflame = pow(flame * .9, 15.0);
    blueflame *= smoothstep(.2, -1.0, _uv.y);
    blueflame /= abs(uv.x * 2.0);
    blueflame = clamp(blueflame, 0.0, 1.0);
    
    // flame
    flame *= smoothstep(1., variationH * 0.5, _uv.y);
    flame = clamp(flame, 0.0, 1.0);
    flame = pow(flame, 3.);
    flame /= smoothstep(1.1, -0.1, _uv.y);    
   
    // colors
    vec4 col = mix(vec4(1.0, 1., 0.0, 0.0), vec4(1.0, 1.0, .6, 0.0), flame);
    col = mix(vec4(1.0, .0, 0.0, 0.0), col, smoothstep(0.0, 1.6, flame));
    fragColor = col;
    
    // a bit blueness on the bottom
    vec4 bluecolor = mix(vec4(0.0, 0.0, 1.0, 0.0), fragColor, 0.95);
    fragColor = mix(fragColor, bluecolor, blueflame);
    
    // clear bg outside of the flame
    fragColor *= flame;
    fragColor.a = flame;
    
    // bg halo
    float haloSize = 0.5;
    float centerL = 1.0 - (length(centerUV + vec2(0.0, 0.1)) / haloSize);
    vec4 halo = vec4(.8, .3, .3, 0.0) * 1.0 * fbm(vec2(iGlobalTime * 0.035)) * centerL + 0.02;
    vec4 finalCol = mix(halo, fragColor, fragColor.a);
    fragColor = finalCol;

    // just a hint of noise
    fragColor *= mix(rand(uv) + rand(uv * .45), 1.0, 0.9);
    fragColor = clamp(fragColor, 0.0, 1.0);
    
}

void main ( void )
{
  mainImage( gl_FragColor, gl_FragCoord.xy );
}